import { NextResponse } from 'next/server';
import prisma from '@/lib/prisma';

export async function GET(
    request: Request,
    { params }: { params: Promise<{ id: string }> } // Next.js 15+ param typing
) {
    try {
        const { id } = await params;
        if (!id) return NextResponse.json({ ok: false, error: 'Missing ID' }, { status: 400 });

        const record = await prisma.serviceRecord.findUnique({
            where: { id },
        });

        if (!record) {
            return NextResponse.json({ ok: false, error: 'Record not found' }, { status: 404 });
        }

        return NextResponse.json({ ok: true, record: record.snapshot });
    } catch (error: any) {
        console.error('Failed to fetch record:', error);
        return NextResponse.json({ ok: false, error: 'Failed to fetch record' }, { status: 500 });
    }
}

export async function PUT(
    request: Request,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        const body = await request.json();
        const { record } = body;

        if (!id || !record) return NextResponse.json({ ok: false, error: 'Invalid data' }, { status: 400 });

        await prisma.serviceRecord.update({
            where: { id },
            data: {
                title: record.title || 'Untitled',
                subjectName: record.details?.name || 'Unknown',
                institution: record.details?.institution,
                term: record.details?.term,
                includedDocs: record.includedDocs || [],
                snapshot: record,
            }
        });

        return NextResponse.json({ ok: true });
    } catch (error: any) {
        console.error('Failed to update record:', error);
        return NextResponse.json({ ok: false, error: 'Failed to update record' }, { status: 500 });
    }
}

export async function DELETE(
    request: Request,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        if (!id) return NextResponse.json({ ok: false, error: 'Missing ID' }, { status: 400 });

        await prisma.serviceRecord.delete({
            where: { id },
        });

        return NextResponse.json({ ok: true });
    } catch (error: any) {
        console.error('Failed to delete record:', error);
        return NextResponse.json({ ok: false, error: 'Failed to delete record' }, { status: 500 });
    }
}
